 
/* ***************************************************
   Pooled Allocation Example
   ***************************************************
   */

#include <iostream.h>
#include <string.h>
#include <stdio.h>
#include <stddef.h>

static void Assert( bool  assertion, char* excuse ) {
     if ( !assertion) { 
         cerr << "Assertion failed: " << excuse << endl;  
     }
 }

class TreeNode {
private:
    enum { BLOCK_SIZE = 10 };
    static void* freeList;

    TreeNode* leftNode;
	TreeNode* rightNode;
	void* data;

public:
    void* TreeNode::operator new( size_t bytesToAllocate );
    void TreeNode::operator delete( void* node, size_t bytesToAllocate );
    static int TreeNode::ItemsFree();

private:
    static void TreeNode::IncreasePool();
    static void TreeNode::AddToFreeList( void* node );
};

/* static */
void* TreeNode::freeList=0;

/* static */
void TreeNode::IncreasePool() {
    char* node = new char[BLOCK_SIZE * sizeof(TreeNode)];
    for( int i=0; i<BLOCK_SIZE; i++)
        AddToFreeList( node + (i * sizeof(TreeNode)) );
}

/* static */
int TreeNode::ItemsFree() {
    int count=0;
    for( void* node = freeList; node != 0; node = *((void**)node) ) 
        count++;
    return count;
}

void* TreeNode::operator new(size_t bytesToAllocate) {
    if( bytesToAllocate != sizeof(TreeNode) ) 
        return ::operator new( bytesToAllocate ); 
    if( freeList == 0)
        IncreasePool();
    void *node = freeList;
    freeList = *((void**)node);
    return node;
}

void TreeNode::operator delete( void* node, size_t bytesToFree ) {
    if( bytesToFree != sizeof(TreeNode) )
        ::operator delete( node );
    else
        AddToFreeList( node );
}

void TreeNode::AddToFreeList( void* node ) {
    *((void**)node) = freeList;
    freeList = node;
}


class NotAListNode : public TreeNode {
    int  i;
};

/* static */
void* TreeNode::freeList=0;

int main( int, char** ) {
    TreeNode* v[20];
    
    Assert( TreeNode::ItemsFree() == 0, "Not 0 items free" );
    for(  int i = 0; i<11; i++)
        v[i] = new TreeNode;
    Assert( TreeNode::ItemsFree() == 9, "Not 9 items free" );

    TreeNode* node = new TreeNode;
    delete node;

    Assert( new TreeNode == node, "Not the same node" );
    delete node;
    for( int j=0; j<i; j++)
        delete v[j];
    Assert( TreeNode::ItemsFree() == 20, "Items lost" );
    NotAListNode* q = new NotAListNode;
    Assert( TreeNode::ItemsFree() == 20, "Derived item taken from list" );
    delete q;
    Assert( TreeNode::ItemsFree() == 20, "Derived item added to list" );

    return 0;
}
