/* Packed Data Examples.
   These produce the following output under MSC:

sizeof(int) == 4
sizeof(bool) == 1
sizeof(short) == 2
sizeof(MessageHeader) == 16
sizeof(ReorderedMessageHeader) == 12
sizeof(PackedMessageHeader) == 9
sizeof(PackedReorderedMessageHeader) == 9
sizeof(BitfieldMessageHeader) == 8
sizeof(BitwiseMessageHeader) == 8
*/

#include <iostream>

using namespace std;

#define DO( x ) { cout << #x << endl; x; }

#pragma pack( 8 )

class MessageHeader {
    bool  inUse;
    int   messageLength;
    char  priority;
    unsigned short channelNumber;
    bool  waitingToSend;
};

class ReorderedMessageHeader {
    int   messageLength;
    unsigned short channelNumber;
    char  priority;
    bool  inUse;
    bool  waitingToSend;
};

#pragma pack( 1 )
class PackedMessageHeader {
    bool  inUse;
    int   messageLength;
    char  priority;
    unsigned short channelNumber;
    bool  waitingToSend;
};

class PackedReorderedMessageHeader {
    int   messageLength;
    unsigned short channelNumber;
    char  priority;
    bool  inUse;
    bool  waitingToSend;
};

class BitfieldMessageHeader {
    int      messageLength;
    unsigned channelNumber: 16;
    unsigned priority:      8;
    unsigned inUse:         1;
    unsigned waitingToSend: 1;
  
public:
    bool IsInUse() { return inUse; }
    void SetInUseFlag( bool isInUse ) { inUse = isInUse; }
    char Priority() { return priority; }
    void SetPriority( char newPriority ) { priority = newPriority; }
    // etc.
};

class BitwiseMessageHeader {
    int      messageLength;
    unsigned short channelNumber;
    char priority;
    unsigned char flags;
public:
    enum FlagName { InUse = 0x01, WaitingToSend = 0x02 };
    bool GetFlag( FlagName f ) { return (flags & f) != 0; }
    void SetFlag( FlagName f ) { flags |= f;  }
    void ResetFlag( FlagName f ) { flags &= ~f;  }
};

#define SHOW( x ) cout << #x " == " << (x) << endl
#define ASSERT( x ) {if (!(x)) cout << "Assertion " #x " failed\n"; }

void main() {
    SHOW( sizeof(int) );
    SHOW( sizeof(bool) );
    SHOW( sizeof(short) );
    SHOW( sizeof(MessageHeader) );
    SHOW( sizeof(ReorderedMessageHeader) );
    SHOW( sizeof(PackedMessageHeader) );
    SHOW( sizeof(PackedReorderedMessageHeader) );
    SHOW( sizeof(BitfieldMessageHeader) );
    SHOW( sizeof(BitwiseMessageHeader) );

    BitwiseMessageHeader x;
    x.SetFlag( BitwiseMessageHeader::InUse );
    x.ResetFlag( BitwiseMessageHeader::WaitingToSend );
    ASSERT( x.GetFlag( BitwiseMessageHeader::InUse) );
    ASSERT(! x.GetFlag( BitwiseMessageHeader::WaitingToSend) );

    SHOW( new MessageHeader );
    for (int i = 0; i < 100; i++)
        ASSERT( ((long)(new MessageHeader) & 0x07 ) == 0 );
}


