
/*
CaptainOates.cpp:
This polls memory regularly.  If memory is low, it chooses the
application lowest in the Z order and sends it a close message.

User Interface:  Only infoprints.  We must use the Notifier, not the
Eikon InfoPrint functionality.  It notifies the user that an
application has terminated. 
*/

#include <apgtask.h>  // TApaTask
#include <e32std.h>   // TTimeIntervalMicroSeconds32 etc.
#include <w32std.h>  // RWsSession

#include <eikenv.h>
#include <eikappui.h>
#include <eikdoc.h>
#include <eikapp.h>
#include <apgtask.h>  // TApaTask
#include <eikcmds.hrh> // EEikCmdExit
#include <e32hal.h>    // UserHal for Memory Use info.

const TUid KUidCaptainOatesApp={203}; // Must correspond to mmp file.


class COatesTerminator : public CBase {
public:                         // Construction and destruction:
    static COatesTerminator* NewL( CEikonEnv* aApplicationEnvironment );
    COatesTerminator( CEikonEnv* aApplicationEnvironment ) :
        iApplicationEnvironment( aApplicationEnvironment ) {} 
    void ConstructL();
    ~COatesTerminator();
  
private:
    void TimerTickL();            // Timer callback mechanism
    static TInt CallbackFunctionL( TAny* );

    TInt GetMemoryPercentFree(); 
    void CloseAnApplicationL();    

    RNotifier  iPopupDialogNotifier;    // Provides user screen output
    CPeriodic* iTimer;                  // Timer mechanism
    CEikonEnv* iApplicationEnvironment; // User I/O Handler for this app.
  
    enum { 
        EPollPeriodInSeconds = 10,      // How often to check memory
        EDangerPercentage = 5 };        // Close applications when less free
                                        // memory than this.
};

COatesTerminator* COatesTerminator::NewL( CEikonEnv* aApplicationEnvironment ) 
{
    COatesTerminator* self =
        new (ELeave) COatesTerminator( aApplicationEnvironment );  
    CleanupStack::PushL( self );
    self->ConstructL();
    CleanupStack::Pop(); // self
    return self;
}

void COatesTerminator::ConstructL()
{
    User::LeaveIfError( iPopupDialogNotifier.Connect() );
    iTimer = CPeriodic::NewL( CActive::EPriorityStandard );
    iTimer->Start( 0, EPollPeriodInSeconds * 1000000L, 
                   TCallBack( CallbackFunctionL, this ) );

    // Set us low priority and in the background :
    RProcess().SetPriority( EPriorityLow );
    iApplicationEnvironment->WsSession().SetWindowGroupOrdinalPosition
        ( iApplicationEnvironment->RootWin().Identifier(), -1 );

    _LIT( KMessage, "Captain Oates application started" );
    iPopupDialogNotifier.InfoPrint( KMessage );
}

COatesTerminator::~COatesTerminator() {
    delete iTimer; // Cancels the timer.
    iPopupDialogNotifier.Close();
}

TInt COatesTerminator::GetMemoryPercentFree() {
    // Gets the notional current memory use, without compressing
    // heaps: 
    TMemoryInfoV1Buf memory;
    UserHal::MemoryInfo(memory);
    return (TInt) ((100L * memory().iFreeRamInBytes) /
                   memory().iTotalRamInBytes); 
}

TInt COatesTerminator::CallbackFunctionL( TAny* self ) {
    STATIC_CAST( COatesTerminator*, self )->TimerTickL();
    return 0;
}

void COatesTerminator::TimerTickL() {
    if ( GetMemoryPercentFree() <= EDangerPercentage ||
         (User::CompressAllHeaps(), 
          GetMemoryPercentFree() <= EDangerPercentage ))  {
        CloseAnApplicationL();
    }
    else {
#ifdef _DEBUG
        _LIT( KMessage, "Free: %d%%" );
        TBuf<20> buf;
        buf.Format( KMessage, GetMemoryPercentFree() );
        iPopupDialogNotifier.InfoPrint( buf );
#endif
    }
}

void COatesTerminator::CloseAnApplicationL() {
    RWsSession& windowServerSession = iApplicationEnvironment->WsSession();
    TInt foregroundApplicationWG = windowServerSession.GetFocusWindowGroup(); 
    TInt systemShellApplicationWG = windowServerSession.GetDefaultOwningWindow();  
    TInt thisApplicationWG = iApplicationEnvironment->RootWin().Identifier(); 
  
    TInt nApplications=windowServerSession.NumWindowGroups(0);
    CArrayFixFlat<TInt>* applicationList=new (ELeave) CArrayFixFlat<TInt>(nApplications);
    CleanupStack::PushL( applicationList );
    User::LeaveIfError( windowServerSession.WindowGroupList(0,applicationList) );
    TInt applicationWG=0;
    TInt i= applicationList->Count();
    for (i--; i>=0; i--)  {
        applicationWG = applicationList->At( i );
        if (applicationWG != thisApplicationWG && 
            applicationWG != systemShellApplicationWG && 
            applicationWG != foregroundApplicationWG) 
            break;
    }
    if (i >= 0) {
        TApaTask task(windowServerSession);
        task.SetWgId(applicationWG);
        task.EndTask();
        _LIT( KMessage, "Application terminated" );
        iPopupDialogNotifier.InfoPrint( KMessage );
    } else {
#ifdef _DEBUG
        _LIT( KMessage, "No terminatable applications" );
        iPopupDialogNotifier.InfoPrint( KMessage );
#endif
    }
    CleanupStack::PopAndDestroy(); // applicationList
}


// *********************** Semi Boiler Plate ********************

//
// COatesAppUi
//

class COatesAppUi : public CEikAppUi
{
public:
    COatesAppUi( CCoeEnv* aCoeEnv ) : iEikEnv( (CEikonEnv*)aCoeEnv ) {}
    void ConstructL();
    ~COatesAppUi();
private: 
    void HandleCommandL(TInt aCommand);
    CEikonEnv* iEikEnv;
    COatesTerminator* iTerminator;
};

void COatesAppUi::ConstructL() {
    BaseConstructL(ENoAppResourceFile);

    iTerminator = new( ELeave ) COatesTerminator( iEikEnv );
    iTerminator->ConstructL();
}

COatesAppUi::~COatesAppUi()
{
    delete iTerminator;
}

void COatesAppUi::HandleCommandL(TInt aCommand)
{
    switch (aCommand)
        {
            //  case EEikCmdFileOpen:
            //          iTerminator->StartTest();
            //          break;
        case EEikCmdExit:  
            Exit();
            break; 
        }
}

//
// COatesDocument
//

class COatesDocument : public CEikDocument
{
public:
    COatesDocument(CEikApplication& aApp,       CCoeEnv* aCoeEnv )
        : CEikDocument(aApp), iCoeEnv( aCoeEnv ) { }
private: // from CApaDocument
    CEikAppUi* CreateAppUiL();
    CCoeEnv* iCoeEnv;
};

CEikAppUi* COatesDocument::CreateAppUiL()
{
    return(new(ELeave) COatesAppUi( iCoeEnv ));
}

//
// COatesApplication
//

class COatesApplication : public CEikApplication
{
private: // from CApaApplication
    CApaDocument* CreateDocumentL();
    TUid AppDllUid() const;
    TFileName ResourceFileName() const;
};

TFileName COatesApplication::ResourceFileName() const
{
    return(TFileName(KNullDesC));
}

TUid COatesApplication::AppDllUid() const
{
    return(KUidCaptainOatesApp);
}

CApaDocument* COatesApplication::CreateDocumentL()
{
    return(new(ELeave) COatesDocument(*this, iCoeEnv));
}

//
// EXPORTed functions
//

EXPORT_C CApaApplication* NewApplication()
{
    return(new COatesApplication);
}

GLDEF_C TInt E32Dll(TDllReason)
{
    return(KErrNone);
}
